<?php
ob_start();
session_start();
header('Content-Type: application/json');

// Error reporting
ini_set('log_errors', 1);
ini_set('error_log', $_SERVER['DOCUMENT_ROOT'] . '/litmac/logs/php_errors.log');
error_reporting(E_ALL);
ini_set('display_errors', 0); // Disable display for production

require $_SERVER['DOCUMENT_ROOT'] . '/litmac/God/Godconn/conn.php';

$response = ['success' => false, 'message' => '', 'redirect' => ''];

try {
    // Strict method checking
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405); // Method Not Allowed
        throw new Exception('Invalid request method. Only POST requests are accepted.');
    }

    // Input validation
    $requiredFields = [
        'company_name' => 'Company name',
        'company_address' => 'Company address',
        'company_email' => 'Company email',
        'company_phone' => 'Company phone',
        'business_category' => 'Business category',
        'username' => 'Username',
        'password' => 'Password'
    ];
    
    foreach ($requiredFields as $field => $fieldName) {
        if (empty($_POST[$field])) {
            throw new Exception("$fieldName is required.");
        }
    }

    // Sanitize inputs
    $companyname = trim(htmlspecialchars($_POST['company_name']));
    $company_address = trim(htmlspecialchars($_POST['company_address']));
    $company_email = filter_var($_POST['company_email'], FILTER_SANITIZE_EMAIL);
    $company_phone = trim($_POST['company_phone']); 
    $username = trim(htmlspecialchars($_POST['username']));
    $password = trim($_POST['password']);
    $business_category = trim($_POST['business_category']);

    // Validate email format
    if (!filter_var($company_email, FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Please enter a valid email address.');
    }

    // Check for existing company, email or username
    $stmt = $conn->prepare("
        SELECT company_name, company_email, username, company_phone
        FROM company 
        WHERE company_name = :companyname 
        OR company_email = :companyemail 
        OR username = :username
        OR company_phone = :companyphone
    ");
    $stmt->bindValue(':companyname', $companyname, PDO::PARAM_STR);
    $stmt->bindValue(':companyemail', $company_email, PDO::PARAM_STR);
    $stmt->bindValue(':username', $username, PDO::PARAM_STR);
    $stmt->bindValue(':companyphone', $company_phone, PDO::PARAM_STR);

    $stmt->execute();

    $existing = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $suggestedUsername = '';

foreach ($existing as $row) {
    if (strtolower($row['company_name']) === strtolower($companyname)) {
        throw new Exception('Company name already exists. Please choose a different name.');
    }

    if (strtolower($row['company_email']) === strtolower($company_email)) {
        throw new Exception('Email address is already registered. Please use a different email or login.');
    }

    if (strtolower($row['company_phone']) === strtolower($company_phone)) {
        throw new Exception('Phone number is already associated with another company.');
    }

    if (strtolower($row['username']) === strtolower($username)) {
        do {
            $suggestedUsername = $username . rand(100, 999);
            $checkStmt = $conn->prepare("SELECT 1 FROM company WHERE username = :suggested");
            $checkStmt->bindValue(':suggested', $suggestedUsername, PDO::PARAM_STR);
            $checkStmt->execute();
        } while ($checkStmt->rowCount() > 0);

        throw new Exception('Username already exists. Suggested username: ' . $suggestedUsername);
    }
}


    // Validate password strength
    if (strlen($password) < 8 || !preg_match('/[A-Z]/', $password) || !preg_match('/[0-9]/', $password)) {
        throw new Exception('Password must be at least 8 characters long, include an uppercase letter and a number.');
    }

    // Hash password
    $passwordHash = password_hash($password, PASSWORD_BCRYPT);
    if (!$passwordHash) {
        throw new Exception('Error hashing password. Please try again.');
    }

    // Insert new company
    $stmt = $conn->prepare("
        INSERT INTO company (
            company_name, 
            company_address, 
            company_email, 
            company_phone, 
            business_category, 
            username, 
            password
        ) VALUES (
            :companyname, 
            :company_address, 
            :company_email, 
            :company_phone, 
            :business_category, 
            :username, 
            :password
        )
    ");

    $stmt->bindValue(':companyname', $companyname, PDO::PARAM_STR);
    $stmt->bindValue(':company_address', $company_address, PDO::PARAM_STR);
    $stmt->bindValue(':company_email', $company_email, PDO::PARAM_STR);
    $stmt->bindValue(':company_phone', $company_phone, PDO::PARAM_STR);
    $stmt->bindValue(':business_category', $business_category, PDO::PARAM_STR);
    $stmt->bindValue(':username', $username, PDO::PARAM_STR);
    $stmt->bindValue(':password', $passwordHash, PDO::PARAM_STR);

    if ($stmt->execute()) {
        $company_id = $conn->lastInsertId();
        $_SESSION['company_id'] = $company_id;

        $response['success'] = true;
        $response['message'] = 'Company account created successfully!';
        $response['redirect'] = "/litmac/authentication/login/login.php";
    } else {
        error_log('Database Error: ' . json_encode($stmt->errorInfo()));
        throw new Exception('Error creating company account. Please try again later.');
    }
} catch (Exception $e) {
    $response['message'] = $e->getMessage();
    error_log("Error: " . $e->getMessage());
}

ob_end_clean();
echo json_encode($response);
exit;